/*
 * dmw-pcm.c -- DMW ASoC platform driver
 *
 *  Copyright (C) 2011 DSPG Technologies GmbH
 *
 * 16 October, 2011:
 * Adapted to kernel 2.6.39 by Avi Miller <Avi.Miller@dspg.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

//#define DEBUG
//#define V_DEBUG

/* CAUTION !!!
 * Defining NRT_DEBUG alters the Real-Time behaviour of DMA operations,
 * due to the printout to console.
 * Use this only for DMA debugging: use "dmesg -n1" to log the code path,
 * and then watch it Offline with "dmesg".
 * Failing to do so, will cause TDM underrun/overrun, which triggers,
 * in the TDM driver, snd_pcm_stop(). This alters the entire sequence.  
 */
//#define NRT_DEBUG

/* GPIOs assist for DMA callbacks debug */
//#define DBG_DMA_GPIOs

/*****************************************************************************/

#include <linux/clk.h>
#include <linux/dmw96dma.h>
#include <linux/firmware.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/vmalloc.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>

#include <sound/dmw-tdm.h>
#include "dmw-pcm.h"

/*****************************************************************************/

#ifdef NRT_DEBUG
	#ifndef DEBUG
		#define DEBUG
	#endif
	#define nrt_dbg_prt(fmt, ...) \
		pr_debug("asoc Pltfrm PCM: %s(): " fmt, __func__, ##__VA_ARGS__)
#else
	#define nrt_dbg_prt(fmt, ...)
#endif


#ifdef V_DEBUG
	#ifndef DEBUG
		#define DEBUG
	#endif
	#define vdbg_prt(fmt, ...) \
		pr_debug("asoc Pltfrm PCM: %s(): " fmt, __func__, ##__VA_ARGS__)
#else
	#define vdbg_prt(fmt, ...)
#endif


#ifdef DEBUG
	#define dbg_prt(fmt, ...) \
		pr_debug("asoc Pltfrm PCM: %s(): " fmt, __func__, ##__VA_ARGS__)
		
	#define DIR_STR(stream_dir)  ((stream_dir == 0) ? "for Playback" : "for Capture")
#else
	#define dbg_prt(fmt, ...)
	#define DIR_SRT(dir)
#endif


#ifdef DBG_DMA_GPIOs
	#include <mach/gpio.h>

	typedef enum {
		kDBG_GPIO_PLBCK_BUF_DONE,
		kDBG_GPIO_CPTUR_BUF_DONE,
		kDBG_GPIO_PLBCK_LIST_FIN,
		kDBG_GPIO_CPTUR_LIST_FIN,
		kDBG_GPIO_PLBCK_XRUN,
		kDBG_GPIO_CPTUR_XRUN,
		kDBG_GPIO_PLBCK_SHIFTWORK,
		kDBG_GPIO_CPTUR_SHIFTWORK,
		kDBG_GPIO_LAST
	} eDBG_GPIO;

	uint g_dbg_gpio[kDBG_GPIO_LAST] = {
		GPIO_PORTB(11), /* Playback Buffer done:	U3CTS  @ EVB96 J15-14 */
		GPIO_PORTB(10), /* Capture  Buffer done:	U3RTS  @ EVB96 J15-16 */
		GPIO_PORTB( 9), /* Playback List finished:	U3RX   @ EVB96 J15-18 */
		GPIO_PORTB( 8), /* Capture  List finished:	U3TX   @ EVB96 J15-20 */

		GPIO_PORTB(15), /* Playback Xrun:		D1RX   @ EVB96 J15-6  */
		GPIO_PORTB(14), /* Capture  Xrun:		T1RX   @ EVB96 J15-8  */
		GPIO_PORTB(13), /* Playback Shift-Work:		F1SYNC @ EVB96 J15-10 */
		GPIO_PORTB(12), /* Capture  Shift-Work:		T1SCLK @ EVB96 J15-12 */
	};
#endif /* DBG_DMA_GPIOs */

/*****************************************************************************/

#define DRV_NAME "dmw-pcm-audio"

/*****************************************************************************/

/* Define one of these compilation switchs to do to "shift" job
 * (PCM buf <--> Interim DMA'ed buf):
 * Define USE_DMA_CALLBACK_FOR_SHIFT_WORK - to be done directly in GDMAC driver's
 *					    callback, called in a TASKLET (!) context.
 *
 * Define USE_WORK_Q_FOR_SHIFT_WORK       - to be done on a private WORK_QUEUE.
 */
#define USE_DMA_CALLBACK_FOR_SHIFT_WORK
//#define USE_WORK_Q_FOR_SHIFT_WORK

#if defined(USE_DMA_CALLBACK_FOR_SHIFT_WORK)
  #undef USE_WORK_Q_FOR_SHIFT_WORK

#elif defined(USE_WORK_Q_FOR_SHIFT_WORK)
  #undef USE_DMA_CALLBACK_FOR_SHIFT_WORK

#else
  #error "Please define way of SHIFT_WORK job"
#endif

/*****************************************************************************/

/* ALSA-pcm-sample to TDM-pcm-sample "conversion" limits, due to TDM HW:
 *   8 bit mono   (ALSA) -> 32 bit stereo (TDM). Not tested, but should work.
 *   8 bit stereo (ALSA) -> 32 bit stereo (TDM). Not tested, but should work.
 *  16 bit mono   (ALSA) -> 32 bit stereo (TDM). Tested ok.
 *  16 bit stereo (ALSA) -> 32 bit stereo (TDM). Currently used and fully tested !
 *  20 bit stereo (ALSA) -> 32 bit stereo (TDM). Not tested, code isn't ready.
 *  24 bit stereo (ALSA) -> 32 bit stereo (TDM). Not tested, code isn't ready.
 *  32 bit stereo (ALSA) -> 32 bit stereo (TDM). Not tested, code isn't ready.
 *
 * Note: TDM HW limitation is 20 meaningful bits out of the 32 bits "TDM sample":
 * dmw96: 20 MSbits; dw74: 20 LSbits.
 */


/* PCM handling limits, reported to ALSA */

/* Note: PCM_PERIOD_BYTES_MIN Must(!) be >= 4 * TDM_FIFO_LVL_FOR_DMA,
 * 	 defined in sound/soc/dmw/dmw-tdm.c, due to basic DMA
 *       transaction size (16 u32-words).
 */
#define PCM_PERIOD_BYTES_MIN    (64) /*   64 Bytes */
#define PCM_PERIOD_BYTES_MAX  (4096) /* 4096 Bytes */

/* Minimal number of Periods in the ALSA PCM buffer */
#define PCM_PERIODS_MIN  (1)

/* Maximal number of Periods in the ALSA PCM buffer:
 * Can be any number greater than PCM_PERIODS_MIN
 * and smaller-equal than (PCM_BUFFER_SIZE_MAX / PCM_PERIOD_BYTES_MIN)
 */
#define PCM_PERIODS_MAX  (1024)

/* Max size of ALSA PCM buffer */
#define PCM_BUFFER_SIZE_MAX  (PCM_PERIOD_BYTES_MIN * PCM_PERIODS_MAX) /* 64 KB */

/* Size of (dynamically allocated) interim double buffers,
 * to be DMA'ed to/from TDM Tx/Rx FIFOs.
 * Worst case is u8-mono PCM to/from u32-stereo over TDM: x8 
 */
#define DMW_PCM_PERIOD_SIZE_TDM  (8 * PCM_PERIOD_BYTES_MAX) /* 32 KB */


/*****************************************************************************/

struct dmw_stream_priv;
struct dmw_pcm_priv;

/* DMA channel states */
typedef enum {

	/* Not running */
	kDMA_ST_IDLE,

	/* Running 'Expicitly', due to explicit ALSA _prepare() callback */
	kDMA_ST_EXPLCT_RUNNING,

	/* Running 'Impicitly', due to _prepar() of Other(!) Direction stream.
	 * Note: Must run dma on other direction as well, due to dmw96's
	 *       TDM machine HW restrictions.
	 */
	kDMA_ST_IMPLCT_RUNNING,

} eDMA_ST;

typedef void (shift_fun_t)(struct dmw_stream_priv *stream, void *p1, void *p2);

/**
 * struct dmw_stream_priv - State information about one play/capt stream
 */
struct dmw_stream_priv {
	uint xfering_pcm; /* Channel is Active, transfering real(!)
			   * PCM samples to/from the ALSA PCM buffer
			   */

	eDMA_ST dma_state; /* DMA channel's 'running' state */

	struct dmw_pcm_priv *pcm;
	struct dmw_stream_priv *other_stream;

	struct snd_pcm_substream *substream;

	uint nun_channels;
	uint bytes_per_channel;
	uint bytes_per_frame;
	size_t buffer_size;

	size_t buffer_bytes;
	snd_pcm_uframes_t period_size;
	size_t period_bytes_alsa;
	size_t period_bytes_tdm;

	int alsa_buffer_idx;

	void *tdm_buffer[2];
	int tdm_buffer_idx_dma;
	int tdm_buffer_idx_sw;

#if defined(USE_WORK_Q_FOR_SHIFT_WORK)
	struct work_struct shift_work;
	uint skip_report_once;
#endif

	shift_fun_t *shift_handler;

	unsigned int hw_channel;
	struct dmw96dma_list *tdm_dma_list;
};

/* struct dmw_pcm_priv - per pcm state */
struct dmw_pcm_priv {

#if defined(USE_WORK_Q_FOR_SHIFT_WORK)
	struct workqueue_struct *wq;
#endif

	/* Marks prevention of DMA/PCM opertaions due to CSS (ARM926)
	 * ownership over the TDM machine due to a DECT phone-call
	 * going on.
	 */
	int tdm_owned_by_css;

	/* Shared lock because both DMA directions are handled simultaneously */
	spinlock_t dma_lock;

	struct snd_card *card;
	struct dmw_stream_priv play;
	struct dmw_stream_priv capt;
};

static const struct snd_pcm_hardware dmw_pcm_hardware = {
	.info			= SNDRV_PCM_INFO_MMAP |
				  SNDRV_PCM_INFO_MMAP_VALID |
				  SNDRV_PCM_INFO_INTERLEAVED |
				  SNDRV_PCM_INFO_BLOCK_TRANSFER |
				  SNDRV_PCM_INFO_BATCH,

	.period_bytes_min	= PCM_PERIOD_BYTES_MIN,
	.period_bytes_max	= PCM_PERIOD_BYTES_MAX,
	.periods_min		= PCM_PERIODS_MIN,
	.periods_max		= PCM_PERIODS_MAX,
	.buffer_bytes_max	= PCM_BUFFER_SIZE_MAX,
};

/*****************************************************************************/

static int g_tdm_owned_by_css = 0;

/*
 * The following set of functions convert the ALSA pcm sample buffers to/from the
 * TDM (32bit, stereo) format.
 */

/* Playback */
/* -------- */

static void play_bit_shift_s8_mono(struct dmw_stream_priv *stream, void *from, void *to)
{
	u8 *in = from;
	u32 *out = to;
	int n = stream->period_size;

	while (n--) {
		*out++ = ((u32)(*in)) << 24;
		*out++ = ((u32)(*in++)) << 24;
	}
}

static void play_bit_shift_s8_stereo(struct dmw_stream_priv *stream, void *from, void *to)
{
	u8 *in = from;
	u32 *out = to;
	int n = stream->period_size;

	while (n--) {
		*out++ = ((u32)(*in++)) << 24;
		*out++ = ((u32)(*in++)) << 24;
	}
}

static void play_bit_shift_s16_mono(struct dmw_stream_priv *stream, void *from, void *to)
{
	u16 *in = from;
	u32 *out = to;
	int n = stream->period_size;

	while (n--) {
		*out++ = ((u32)(*in)) << 16;
		*out++ = ((u32)(*in++)) << 16;
	}
}

static void play_bit_shift_s16_stereo(struct dmw_stream_priv *stream, void *from, void *to)
{
	u16 *in = from;
	u32 *out = to;
	int n = stream->period_size;

	while (n--) {
		*out++ = ((u32)(*in++)) << 16;
		*out++ = ((u32)(*in++)) << 16;
	}
}

/* Capture */
/* ------- */

static void capt_bit_shift_s8_mono(struct dmw_stream_priv *stream, void *to, void *from)
{
	u32 *in = from;
	u8 *out = to;
	int n = stream->period_size;

	while (n--) {
		*out++ = (u8)(*in >> 24);
		in += 2;
	}
}

static void capt_bit_shift_s8_stereo(struct dmw_stream_priv *stream, void *to, void *from)
{
	u32 *in = from;
	u8 *out = to;
	int n = stream->period_size;

	while (n--) {
		*out++ = (u8)(*in++ >> 24);
		*out++ = (u8)(*in++ >> 24);
	}
}

static void capt_bit_shift_s16_mono(struct dmw_stream_priv *stream, void *to, void *from)
{
	u32 *in = from;
	u16 *out = to;
	int n = stream->period_size;

	while (n--) {
		*out++ = (u16)(*in >> 16);
		in += 2;
	}
}

static void capt_bit_shift_s16_stereo(struct dmw_stream_priv *stream, void *to, void *from)
{
	u32 *in = from;
	u16 *out = to;
	int n = stream->period_size;

	while (n--) {
		*out++ = (u16)(*in++ >> 16);
		*out++ = (u16)(*in++ >> 16);
	}
}

static shift_fun_t *get_shift_fun(int stream, struct snd_pcm_hw_params *params)
{
	/* [stream][channels][format] */
	static shift_fun_t * const funs[SNDRV_PCM_STREAM_LAST+1][2][2] = {
		[SNDRV_PCM_STREAM_PLAYBACK] = {
			{play_bit_shift_s8_mono, play_bit_shift_s16_mono},
			{play_bit_shift_s8_stereo, play_bit_shift_s16_stereo},
		},
		[SNDRV_PCM_STREAM_CAPTURE] = {
			{capt_bit_shift_s8_mono, capt_bit_shift_s16_mono},
			{capt_bit_shift_s8_stereo, capt_bit_shift_s16_stereo},
		},
	};
	unsigned int fmt;
	unsigned int channels = params_channels(params) - 1;

	if (channels > 1) {
		return NULL;
	}

	switch (params_format(params)) {
		case SNDRV_PCM_FORMAT_S8:
			fmt = 0;
			break;
		case SNDRV_PCM_FORMAT_S16:
			fmt = 1;
			break;
		default:
			return NULL;
	}

	return funs[stream][channels][fmt];
}

/*****************************************************************************/

/* handle_xrun - Handle DMA under-/overflow
 *
 * Handles a DMA under-/overflow by stopping the stream and informing the
 * user space about the XRUN.
 */
static void handle_xrun(struct dmw_stream_priv *stream)
{
	unsigned long flags;
	unsigned int pcm_stopped = 0;

#ifdef DBG_DMA_GPIOs
	if (stream->substream) {
		gpio_set_value( (stream->substream->stream) ?
					g_dbg_gpio[kDBG_GPIO_PLBCK_XRUN] :
					g_dbg_gpio[kDBG_GPIO_CPTUR_XRUN],
				1 );
	}
#endif

	printk(KERN_NOTICE "handle_xrun: DMA underrun/overrun !!!");

	snd_pcm_stream_lock_irqsave(stream->substream, flags);
	if (stream->xfering_pcm) {
		pcm_stopped = 1;
		snd_pcm_stop(stream->substream, SNDRV_PCM_STATE_XRUN);
	}
	snd_pcm_stream_unlock_irqrestore(stream->substream, flags);

	if (pcm_stopped) {
		nrt_dbg_prt("%s %s\n",
			(stream->substream)? DIR_STR(stream->substream->stream) : "?",
			"pcm stream Stopped with PCM_STATE_XRUN");
	}

#ifdef DBG_DMA_GPIOs
	if (stream->substream) {
		gpio_set_value( (stream->substream->stream) ?
					g_dbg_gpio[kDBG_GPIO_PLBCK_XRUN] :
					g_dbg_gpio[kDBG_GPIO_CPTUR_XRUN],
				0 );
	}
#endif
}

/* xfer_shift_work - Convert one period To (on Playbak) or
 * From (on Capture) interim DMA'ed buffer.
 *
 * Calls the bit-shift function for the next period if the stream is still
 * active. Then the buffer pointer is updated and ALSA is informed that a
 * period has been consumed.
 */

#if defined(USE_WORK_Q_FOR_SHIFT_WORK)
  static void xfer_shift_work(struct work_struct *work)
#elif defined(USE_DMA_CALLBACK_FOR_SHIFT_WORK)
  static void xfer_shift_work(struct dmw_stream_priv *stream)
#endif
{

#if defined(USE_WORK_Q_FOR_SHIFT_WORK)
	struct dmw_stream_priv *stream =
		container_of(work, struct dmw_stream_priv, shift_work);
#endif

	void *period_buf;

#ifdef DBG_DMA_GPIOs
	if (stream->substream) {
		gpio_set_value( (stream->substream->stream) ?
					g_dbg_gpio[kDBG_GPIO_PLBCK_SHIFTWORK] :
					g_dbg_gpio[kDBG_GPIO_CPTUR_SHIFTWORK],
				1 );
	}
#endif

	if (!stream->xfering_pcm) {
		return;
	}

	/* Process one Period */
	period_buf = stream->substream->runtime->dma_area +
		stream->alsa_buffer_idx;
	stream->shift_handler(stream, period_buf,
		stream->tdm_buffer[stream->tdm_buffer_idx_sw]);

	/* Flip to other tdm-buffer */
	stream->tdm_buffer_idx_sw ^= 0x01;

	/* finally consume period */

	/* Note: We are assured to handle (one single) Full Period at a time,
	 * due to "snd_pcm_hw_constraint_integer(runtime, SNDRV_PCM_HW_PARAM_PERIODS)"
	 * in dmw_pcm_open()
	 */
	stream->alsa_buffer_idx += stream->period_bytes_alsa;
	if (stream->alsa_buffer_idx >= stream->buffer_bytes) {
		stream->alsa_buffer_idx = 0;
	}

#ifdef DBG_DMA_GPIOs
	if (stream->substream) {
		gpio_set_value( (stream->substream->stream) ?
					g_dbg_gpio[kDBG_GPIO_PLBCK_SHIFTWORK] :
					g_dbg_gpio[kDBG_GPIO_CPTUR_SHIFTWORK],
				0 );
	}
#endif
}

/* tdm_dma_buf_done - DMA buffer done callback
 *
 * Called by the DMA driver every time a buffer has been finished. Used to kick
 * off the bit shifter work to re-fill this buffer.
 */
static void tdm_dma_buf_done(int status, void *context)
{
	struct dmw_stream_priv *strm = context;

#ifdef DBG_DMA_GPIOs
	if (strm->substream) {
		gpio_set_value( (strm->substream->stream) ?
					g_dbg_gpio[kDBG_GPIO_PLBCK_BUF_DONE] :
					g_dbg_gpio[kDBG_GPIO_CPTUR_BUF_DONE],
				1 );
	}
#endif

	/* Ignore errors. Mostly due to Legal dma transfer canceling.
	 * The tdm_dma_list_finished() handler, for dma-list ending, will catch it.
	 */
	if (unlikely(status)) {
		nrt_dbg_prt("%s %s %s %i%s\n",
			 "DMA",
			 (strm->substream) ? DIR_STR(strm->substream->stream) : "?",
			 "Period Buf finished with err",
			 status, ". Ignoring");

		goto L_RETURN;
	}

	/* Flip DMA buffer */
	strm->tdm_buffer_idx_dma ^= 0x01;

	/* Note: if stream is not Active, passing PCM data from/to the alsa buffer,
	 * there is no need to perform the "shift" work. Just return.
	 */
	if (!strm->xfering_pcm) {

		goto L_RETURN;
	}

	/* DMA has moved to the next buffer. Trigger the bit-shifter
	 * and report progress.
	 * Otherwise, if no new data is available, do under/over-run recovery.
	 */
	if (likely(strm->tdm_buffer_idx_dma != strm->tdm_buffer_idx_sw)) {

#if defined(USE_WORK_Q_FOR_SHIFT_WORK)
		queue_work(strm->pcm->wq, &strm->shift_work);

		/* Skip pcm progress report at Fisrt 'dma buf-done' -
		 * it is Async to the 'Shift-Work' where wq is used.
		 */
		if (unlikely(strm->skip_report_once)) {
			strm->skip_report_once = 0;

			goto L_RETURN;
		}
#elif defined(USE_DMA_CALLBACK_FOR_SHIFT_WORK)
		xfer_shift_work(strm);
#endif

		/* Notify ALSA (under tasklet context !) about pcm-buffer
		 * read/write progress (1 Period)
		 */
		snd_pcm_period_elapsed(strm->substream);

	} else {
		printk(KERN_NOTICE "%s\n", "No 'Shift-work' on time: Forcing XRUN !");
		handle_xrun(strm);
	}


L_RETURN:
#ifdef DBG_DMA_GPIOs
	if (strm->substream) {
		gpio_set_value( (strm->substream->stream) ?
					g_dbg_gpio[kDBG_GPIO_PLBCK_BUF_DONE] :
					g_dbg_gpio[kDBG_GPIO_CPTUR_BUF_DONE],
				0 );
	}
#endif

	return;
}

/* tdm_dma_list_finished - DMA list finished callback
 *
 * Called by the DMA driver when the DMA list has been finished.
 * Should only be called if the DMA was canceled (status == -EINTR)
 * or if the DMA driver could not keep up with the hardware
 * thus the circular DMA list got stopped.
 */
static void tdm_dma_list_finished(int status, void *context)
{
	struct dmw_stream_priv *stream = context;
	unsigned long flags;
	unsigned int pcm_stopped = 0;

#ifdef DBG_DMA_GPIOs
	if (stream->substream) {
		gpio_set_value( (stream->substream->stream) ?
					g_dbg_gpio[kDBG_GPIO_PLBCK_LIST_FIN] :
					g_dbg_gpio[kDBG_GPIO_CPTUR_LIST_FIN],
				1 );
	}
#endif

	if (status == -EINTR) {
		/* Note: DMA has been Canceled */
		snd_pcm_stream_lock_irqsave(stream->substream, flags);
		if (unlikely(stream->xfering_pcm)) {
			pcm_stopped = 1;
			snd_pcm_stop(stream->substream, SNDRV_PCM_STATE_DISCONNECTED);
		}
		snd_pcm_stream_unlock_irqrestore(stream->substream, flags);

		if (unlikely(pcm_stopped)) {
			nrt_dbg_prt("%s %s\n",
				(stream->substream)?
					DIR_STR(stream->substream->stream) : "?",
				"pcm stream Stopped with PCM_STATE_DISCONNECTED");
		} else {
			nrt_dbg_prt("%s %s %s\n",
				 "DMA list",
				 (stream->substream) ?
					DIR_STR(stream->substream->stream) : "?",
				 "canceled (OK)");
		}

		goto L_RETURN;
	}

	/* Note: the system could not keep the DMA running! */

	nrt_dbg_prt("%s %s %s %i\n",
		"DMA list",
		(stream->substream) ? DIR_STR(stream->substream->stream) : "?",
		"finished with err:",
		status);

	if (stream->xfering_pcm) {
		nrt_dbg_prt("%s\n", "	while stream was Active --> Forcing XRUN !");
		handle_xrun(stream);
	}

	/* Just too slow or something really bad? */
	if (status == -EPIPE) {
		/* Too slow -> just re-submit */
		stream->tdm_buffer_idx_dma = 0;
		if (!(status = dmw96dma_submit(stream->tdm_dma_list))) {
			nrt_dbg_prt("%s\n", "	EPIPE: too slow --> Resubmitted !");

			goto L_RETURN;
		}
		nrt_dbg_prt("%s\n", "Resubmit Failed !");
	}

	/* Huuh, got a hard DMA error. The DMA for this direction is gone. If
	 * the TDM is still active, then the FIFO will over/under-run and the
	 * TDM will finally be stopped!
	 */
	dev_err(stream->pcm->card->dev,
		"tdm_dma_list_finished(): hard DMA error %d\n", status);

L_RETURN:
#ifdef DBG_DMA_GPIOs
	if (stream->substream) {
		gpio_set_value( (stream->substream->stream) ?
					g_dbg_gpio[kDBG_GPIO_PLBCK_LIST_FIN] :
					g_dbg_gpio[kDBG_GPIO_CPTUR_LIST_FIN],
				0 );
	}
#endif

	return;
}

/* tdm_dma_configure - Configure TDM DMA chain
 *
 * The circular TDM DMA list for this stream is (re-)created. If the DMA is
 * already prepared, that is the transfer is already running because the other
 * direction is active, we will swap the transfer with the new and make sure
 * its running.
 */
static int tdm_dma_configure(struct dmw_stream_priv *stream)
{
	struct dmw96dma_list *new_dma_list;
	struct dmw96dma_list *old_dma_list;
	int ret;

	dbg_prt("%s\n",
		(stream->substream)? DIR_STR(stream->substream->stream) : "?");

	/* Allocate New (circular) DMA list and add it the two DMA buffers */
	new_dma_list = dmw96dma_alloc_io_list(1, stream->hw_channel,
		&tdm_dma_list_finished, stream);
	if (!new_dma_list) {
		return -ENOMEM;
	}

	ret = dmw96dma_add_io_transfer_buf(new_dma_list, stream->tdm_buffer[0],
			stream->period_bytes_tdm, &tdm_dma_buf_done, stream);
	if (ret) {
		goto L_FREE_LIST;
	}

	ret = dmw96dma_add_io_transfer_buf(new_dma_list, stream->tdm_buffer[1],
			stream->period_bytes_tdm, &tdm_dma_buf_done, stream);
	if (ret) {
		goto L_FREE_LIST;
	}


	/* Check if Old DMA list exists */
	if (stream->tdm_dma_list) {
		/* Temp save the Old DMA list */
		old_dma_list = stream->tdm_dma_list;
		/* Save the New DMA list */
		stream->tdm_dma_list = new_dma_list;

		if (stream->dma_state != kDMA_ST_IDLE) {

			spin_lock_bh(&stream->pcm->dma_lock);

			/* Immediately submit the New DMA List,
			 * because the old one was already running.
			 * (Not changing its current DMA state).
			 *
			 * This is a must(!) for the TDM machine, in case
			 * the other direction's DMA is running.
			 */
			stream->tdm_buffer_idx_dma = 0;
			ret = dmw96dma_submit(new_dma_list);
			if (ret == 0) {
				nrt_dbg_prt("%s\n",
					"Old DMA list Replaced. New DMA list Submitted OK");
			} else {
				/* WTF? Well, the TDM FIFO will under/over-run.
				 * Will be handled by TDM driver.
				 */
				stream->dma_state = kDMA_ST_IDLE;
				dev_err(stream->pcm->card->dev,
					"%s: New DMA submit err %d\n", __FUNCTION__, ret);
			}

			spin_unlock_bh(&stream->pcm->dma_lock);

			/* Gracefully Stop current DMA transfer.
			 * Note: returns when the DMA Burst(!) finishes.
			 */
			dmw96dma_cancel(old_dma_list);
		}

		/* Free Old DMA list */
		dmw96dma_free_list(old_dma_list);
		nrt_dbg_prt("%s\n", "Old DMA list Freed");

	} /* end_if Old DMA list exists */

	else {
		/* Save the New DMA list */
		stream->tdm_dma_list = new_dma_list;
		stream->dma_state = kDMA_ST_IDLE;
	}

	return ret;

L_FREE_LIST:
	dmw96dma_free_list(new_dma_list);
	return ret;
}

static int tdm_dma_prepare_locked(struct dmw_stream_priv *stream)
{
	int ret = 0;
	struct dmw_stream_priv *other_stream = NULL;

	dbg_prt("%s\n",
		(stream->substream)? DIR_STR(stream->substream->stream) : "?");

	stream->tdm_buffer_idx_dma = 0;

	switch (stream->dma_state) {
	case kDMA_ST_IDLE:
		/* Submit the channel's DMA List */
		ret = dmw96dma_submit(stream->tdm_dma_list);
		if (ret == 0) {
			stream->dma_state = kDMA_ST_EXPLCT_RUNNING;
			vdbg_prt("%s %s %s\n",
				"DMA list",
				(stream->substream) ?
					DIR_STR(stream->substream->stream) : "?",
				"submitted OK");
		} else {
			vdbg_prt("%s %s %s %i\n",
				"DMA list",
				(stream->substream) ?
					DIR_STR(stream->substream->stream) : "?",
				"submit Failed ! err:",
				ret);

			/* Exit badly */
			return ret;
		}
		break;
	case kDMA_ST_IMPLCT_RUNNING:
		/* Already submitted internally - No need to re-submit */
		stream->dma_state = kDMA_ST_EXPLCT_RUNNING;
		break;
	case kDMA_ST_EXPLCT_RUNNING:
		/* Do nothing */
		break;
	default:
		vdbg_prt("%s\n","Unexpected DMA state");
		break;
	}

	/* Note: due to the DMW96 TDM machine HW capabilities,
	 *	 dma Must run DMA on both ways
	 */ 

	other_stream = stream->other_stream;
	if (!other_stream) {
		return ret;
	}

	switch (other_stream->dma_state) {
	case kDMA_ST_IDLE:
		other_stream->tdm_buffer_idx_dma = 0;
		ret = dmw96dma_submit(other_stream->tdm_dma_list);
		if (ret == 0) {
			other_stream->dma_state = kDMA_ST_IMPLCT_RUNNING;
			vdbg_prt("%s %s %s\n",
				"DMA list (other direction)",
				(other_stream->substream) ?
					DIR_STR(other_stream->substream->stream) : "?",
				"submitted OK as well ");
		} else {
			vdbg_prt("%s %s %s\n",
				"DMA list (other direction)",
				(other_stream->substream) ?
					DIR_STR(other_stream->substream->stream) : "?",
				"submit Failed !! Stopping DMA on Orig direction too");

			/* Stop DMA transfer of the Original(!) stream */
			dmw96dma_cancel(stream->tdm_dma_list);
			stream->dma_state = kDMA_ST_IDLE;
		}
		break;

	case kDMA_ST_IMPLCT_RUNNING:
	case kDMA_ST_EXPLCT_RUNNING:
		/* Do Nothing */
		break;

	default:
		vdbg_prt("%s\n","Unexpected DMA state");
		break;
	}

	return ret;
}

/* tdm_dma_prepare - Prepare (submit) DMA transfer into/from TDM FIFO.
 *
 * Submit DMA transfer.
 * Note: submits the other direction as well,
 * because the TDM machine needs both directions simultaneously.
 */
static int tdm_dma_prepare(struct dmw_stream_priv *stream)
{
	int ret = 0;

	dbg_prt("%s\n",
		(stream->substream)? DIR_STR(stream->substream->stream) : "?");

	spin_lock_bh(&stream->pcm->dma_lock);

	if (stream->dma_state != kDMA_ST_EXPLCT_RUNNING) {
		ret = tdm_dma_prepare_locked(stream);
	}

	spin_unlock_bh(&stream->pcm->dma_lock);

	return ret;
}

/* tdm_dma_cancel_xfer - Cancel DMA transfer (if allowed to).
 *
 * Note: cancels the other direction as well, only if not
 * currently streaming to/from ALSA PCM buffer.
 */
static void tdm_dma_cancel_xfer(struct dmw_stream_priv *stream)
{
	dbg_prt("%s\n",
		(stream->substream)? DIR_STR(stream->substream->stream) : "?");

	if (!stream->pcm) {
		vdbg_prt("%s\n", "No pcm. Aborting");
		return;
	}

	spin_lock_bh(&stream->pcm->dma_lock);

	/* Check if This stream is still Active -
	 * didn't get SNDRV_PCM_TRIGGER_STOP/SUSPEND/PAUSE_PUSH trigger
	 */
	if (stream->xfering_pcm) {
		vdbg_prt("%s\n", "Stream is still Active. Ignoring");
		goto L_UNLOCK;
	}

	if (stream->dma_state == kDMA_ST_IDLE) {
		vdbg_prt("%s\n", "DMA channel state is IDLE. Doing nothing");
		goto L_UNLOCK;
	}

	/* Check if Other-direction stream is Active or has DMA running */
	if ((stream->dma_state == kDMA_ST_EXPLCT_RUNNING) &&
	    ( (stream->other_stream->xfering_pcm) ||
	      (stream->other_stream->dma_state == kDMA_ST_EXPLCT_RUNNING) )) {
		/* Can't stop This-direction DMA for now.
		 * Just mark This stream as running Implicitly.
		 * 
		 * DMA will be stopped on Other-direction stream's
		 * DMA canceling.
		 */
		stream->dma_state = kDMA_ST_IMPLCT_RUNNING;
		vdbg_prt("%s\n", "Other direction is still Running. Not canceling");
		goto L_UNLOCK;
	}

	/* Stop channel's DMA xfer */
	dmw96dma_cancel(stream->tdm_dma_list);
	stream->dma_state = kDMA_ST_IDLE;

	vdbg_prt("%s %s %s\n",
		"DMA transfer",
		(stream->substream) ?
			DIR_STR(stream->substream->stream) : "?",
		"Cancled OK.");

	/* Stop Other-direction channel's DMA xfer, if allowed */
	if (stream->other_stream->dma_state != kDMA_ST_EXPLCT_RUNNING) {
		dmw96dma_cancel(stream->other_stream->tdm_dma_list);
		stream->other_stream->dma_state = kDMA_ST_IDLE;

		vdbg_prt("%s %s %s\n",
			"DMA transfer (other direction)",
			(stream->other_stream->substream) ?
				DIR_STR(stream->substream->stream) : "?",
			"Cancled as well.");
	}

L_UNLOCK:
	spin_unlock_bh(&stream->pcm->dma_lock);
}

/* dmw_pcm_start - Start substream
 *
 * The DMA is already running. To start playback/capture, mark the stream
 * as Active. This will kick of the 'Shift Work' to consume the first period.
 */
static int dmw_pcm_start(struct dmw_stream_priv *stream)
{
	dbg_prt("%s\n",
		(stream->substream)? DIR_STR(stream->substream->stream) : "?");

	if (!stream->pcm->tdm_owned_by_css) {
		snd_BUG_ON(stream->dma_state == kDMA_ST_IDLE);
		stream->tdm_buffer_idx_sw = stream->tdm_buffer_idx_dma;
		stream->xfering_pcm = 1;

#ifdef USE_WORK_Q_FOR_SHIFT_WORK
		stream->skip_report_once = 1;
#endif

		return 0;

	} else {
		vdbg_prt("%s\n", "TDM owned by CSS. Aborting!");
		return -EPERM;
	}
}

/* dmw_pcm_stop - Stop substream
 *
 * To stop playback/capture, mark the stream as InActive.
 * Keep the DMA running! This prevents a non-recoverable TDM FIFO
 * under/over-run in This direction, if the Other direction is still active.
 *
 * Playback will keep on filling the TDM TX fifo with dummy all-'0' spamples .
 * Capture will keep on emptying the TDM RX fifo. 
 */
static int dmw_pcm_stop(struct dmw_stream_priv *stream)
{
	dbg_prt("%s\n",
		(stream->substream)? DIR_STR(stream->substream->stream) : "?");

	stream->xfering_pcm = 0;
	memset(stream->tdm_buffer[0], 0, DMW_PCM_PERIOD_SIZE_TDM);
	memset(stream->tdm_buffer[1], 0, DMW_PCM_PERIOD_SIZE_TDM);

	return 0;
}

/*****************************************************************************/

 /*
 * ALSA PCM callbacks
 */

int dmw_pcm_open(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct dmw_pcm_priv *pcm_priv = snd_soc_dai_get_pcm_privdata(rtd->cpu_dai);

	struct dmw_stream_priv *stream_priv;
	int ret;

	dbg_prt("%s\n", DIR_STR(substream->stream));

	/* Set our PCM capabilities */
	snd_soc_set_runtime_hwparams(substream, &dmw_pcm_hardware);

	/* Add DMA constraint: Peroid Size, in bytes,
	 * must be a multiple of the DMA burst size: 16 u32's = 64,
	 * as interpreted in the ALSA PCM buffer:
	 * 1 "DMA" u32 mono-sample (4 Bytes) <--> 1 "PCM" u16 mono-sample (2 Bytes),
	 * which is 64 / 2 = 32
	 */
	ret = snd_pcm_hw_constraint_step(runtime, 0,
		SNDRV_PCM_HW_PARAM_PERIOD_BYTES, PCM_PERIOD_BYTES_MIN / 2); /* 32 bytes */
	if (ret) {
		return ret;
	}

	/* Add DMA constraint: PCM buf logical Size, in bytes,
	 * must be a multiple of the DMA burst size: 16 u32's = 64,
	 * as interpreted in the ALSA PCM buffer:
	 * 1 "DMA" u32 mono-sample (4 Bytes) <--> 1 "PCM" u16 mono-sample (2 Bytes),
	 * which is 64 / 2 = 32.
	 */
	ret = snd_pcm_hw_constraint_step(runtime, 0,
		SNDRV_PCM_HW_PARAM_BUFFER_BYTES, PCM_PERIOD_BYTES_MIN / 2); /* 32 bytes */
	if (ret) {
		return ret;
	}

	/* Add PCM constraint: 
	 * ALSA PCM buffer must hold integral number of Peroids
	 */
	ret = snd_pcm_hw_constraint_integer(runtime,
		SNDRV_PCM_HW_PARAM_PERIODS);
	if (ret < 0) {
		return ret;
	}

	/* Set runtime structure */
	stream_priv = (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) ?
			&pcm_priv->play : &pcm_priv->capt;
	stream_priv->xfering_pcm = 0;
	stream_priv->substream = substream;
	runtime->private_data = stream_priv;

	return 0;
}

static int dmw_pcm_close(struct snd_pcm_substream *substream)
{
	struct dmw_stream_priv *stream_priv = substream->runtime->private_data;

	dbg_prt("%s\n", DIR_STR(substream->stream));

	/* Cancle DMA transfer */
	tdm_dma_cancel_xfer(stream_priv);

#if defined(USE_WORK_Q_FOR_SHIFT_WORK)
	/* Clean up workqueue */
	cancel_work_sync(&stream_priv->shift_work);
#endif

	/* Free ALSA pcm buffer */
	snd_pcm_lib_free_vmalloc_buffer(substream);

	stream_priv->substream = NULL;

	return 0;
}

static int dmw_pcm_hw_params(struct snd_pcm_substream *substream,
	struct snd_pcm_hw_params *params)
{
	struct dmw_stream_priv *stream_priv = substream->runtime->private_data;
	size_t buffer_bytes = params_buffer_bytes(params);
	int ret;

	dbg_prt("%s\n", DIR_STR(substream->stream));

#if defined(USE_WORK_Q_FOR_SHIFT_WORK)
	/* Clean up workqueue */
	cancel_work_sync(&stream_priv->shift_work);
#endif

	/* Allocate ALSA PCM buffer */
	ret = snd_pcm_lib_alloc_vmalloc_buffer(substream, buffer_bytes);
	if (ret < 0) {
		return ret;
	}

	/* Save the buffer and period sizes for the processing */
	stream_priv->buffer_bytes = buffer_bytes;

	/* Num of Frames(!) in Peroid, in ALSA PCM buf */
	stream_priv->period_size = params_period_size(params);
	stream_priv->period_bytes_alsa = params_period_bytes(params);

	stream_priv->period_bytes_tdm = stream_priv->period_size * 8; /* correct for any kind of Frame ! */
	stream_priv->alsa_buffer_idx = 0;

	stream_priv->buffer_size = params_buffer_size(params);
	vdbg_prt("%s %u %s  %s %u %s\n",
		 "PCM buffer_size:",  stream_priv->buffer_size, "frames.",
		 "PCM buffer_bytes:", stream_priv->buffer_bytes, "bytes");

	vdbg_prt("%s %u %s  %s %u %s\n",
		 "Period size:",  (uint)stream_priv->period_size, "frames.",
		 "Period bytes:", stream_priv->period_bytes_alsa, "bytes");

	stream_priv->nun_channels = params_channels(params);
	stream_priv->bytes_per_channel =
		(params_format(params) == SNDRV_PCM_FORMAT_S16) ? 2 : 1;
	stream_priv->bytes_per_frame =
		stream_priv->nun_channels * stream_priv->bytes_per_channel;
	vdbg_prt("%s %u  %s %u  %s %u\n",
		 "Num Channels:",  stream_priv->nun_channels,
		 "Bytes-per-Ch:", stream_priv->bytes_per_channel,
		 "Bytes-per-Frame:" , stream_priv->bytes_per_frame);

	/* Bit-shift function */
	stream_priv->shift_handler = get_shift_fun(substream->stream, params);
	if (!(stream_priv->shift_handler)) {
		return -EINVAL;
	}

	/* Initialize TDM DMA */
	vdbg_prt("%s %s\n",
		 "Actual config DMA chain",
		 (substream) ? DIR_STR(substream->stream) : "?");
	ret = tdm_dma_configure(stream_priv);
	if (ret) {
		return ret;
	}

	return 0;
}

static int dmw_pcm_hw_free(struct snd_pcm_substream *substream)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));

	vdbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;	
}

static int dmw_pcm_prepare(struct snd_pcm_substream *substream)
{
	struct dmw_stream_priv *stream = substream->runtime->private_data;
	int ret;

	dbg_prt("%s\n", DIR_STR(substream->stream));

	/* Save the Dynamic 'Telephony' state on Current substream */
	stream->pcm->tdm_owned_by_css = g_tdm_owned_by_css;

	if (!stream->pcm->tdm_owned_by_css) {

		/* Clear the DMA buffers */
		memset(stream->tdm_buffer[0], 0, DMW_PCM_PERIOD_SIZE_TDM);
		memset(stream->tdm_buffer[1], 0, DMW_PCM_PERIOD_SIZE_TDM);
		stream->alsa_buffer_idx = 0;

		/* Launch the DMA channel */
		ret = tdm_dma_prepare(stream);
		if (ret) {
			return ret;
		}

	} else {
		vdbg_prt("%s\n", "TDM owned by CSS");
		if (stream->dma_state != kDMA_ST_IDLE) {
			vdbg_prt("%s\n", "DMA ch is running: Canceling DMA xfer");
			/* Cancel DMA transfer */
			tdm_dma_cancel_xfer(stream);
		} else {
			vdbg_prt("%s\n", "DMA ch is idle: Doing nothing");
		}

	}

	return 0;
}

static int dmw_pcm_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct dmw_stream_priv *stream = substream->runtime->private_data;
	int ret = -EINVAL;

	dbg_prt("%s\n", DIR_STR(substream->stream));

	switch (cmd) {
		case SNDRV_PCM_TRIGGER_START:
		case SNDRV_PCM_TRIGGER_RESUME:
		case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
			vdbg_prt("%s\n", "Start/Resume/Pause-Release trigger");
			if (!stream->pcm->tdm_owned_by_css) {
				vdbg_prt("%s\n", "TDM owned by Cortex. Starting PCM xfer vs. ALSA!");
				ret = dmw_pcm_start(stream);
			} else {
				snd_pcm_stop(substream, SNDRV_PCM_STATE_DISCONNECTED);
				vdbg_prt("%s\n", "TDM owned by CSS. Not starting PCM xfer vs. ALSA!");
			}
			break;

		case SNDRV_PCM_TRIGGER_STOP:
		case SNDRV_PCM_TRIGGER_SUSPEND:
		case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
			vdbg_prt("%s\n", "Stop/Suspend/Pause-Push trigger");
			if (!stream->pcm->tdm_owned_by_css) {
				vdbg_prt("%s\n", "TDM owned by Cortex");
			} else {
				vdbg_prt("%s\n", "TDM owned by CSS");
			}
			vdbg_prt("%s\n", "Stopping PCM xfer vs. ALSA!");
			ret = dmw_pcm_stop(stream);
			break;

		default:
			vdbg_prt("%s %i\n", "Un handled pcm trigger:", cmd);
	}

	return ret;
}

static snd_pcm_uframes_t dmw_pcm_pointer(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct dmw_stream_priv *stream = runtime->private_data;

	return bytes_to_frames(runtime, stream->alsa_buffer_idx);
}

static struct snd_pcm_ops dmw_pcm_ops = {
	.open		= dmw_pcm_open,
	.close		= dmw_pcm_close,
	.ioctl		= snd_pcm_lib_ioctl,
	.hw_params	= dmw_pcm_hw_params,
	.hw_free	= dmw_pcm_hw_free,
	.prepare	= dmw_pcm_prepare,
	.trigger	= dmw_pcm_trigger,
	.pointer	= dmw_pcm_pointer,
	.copy		= NULL,
	.mmap		= snd_pcm_lib_mmap_vmalloc,
	.silence	= NULL,
	.page		= snd_pcm_lib_get_vmalloc_page,
	.ack		= NULL,
};

/*****************************************************************************/

static void dmw_pcm_free_dma_data(struct dmw_stream_priv *stream)
{
	dbg_prt("%s\n",
		(stream->substream)? DIR_STR(stream->substream->stream) : "?");

	if (stream->tdm_buffer[0]) {
		kfree(stream->tdm_buffer[0]);
	}
	if (stream->tdm_buffer[1]) {
		kfree(stream->tdm_buffer[1]);
	}

	if (stream->tdm_dma_list) {
		dmw96dma_free_list(stream->tdm_dma_list);
	}
}

static int dmw_pcm_new(struct snd_card *card, struct snd_soc_dai *codec_dai,
			struct snd_pcm *pcm)
{
	struct snd_soc_pcm_runtime *rtd = pcm->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	struct dmw_pcm_priv *pcm_priv;
	int ret = -ENOMEM;

	dbg_prt();

	pcm_priv = kzalloc(sizeof(*pcm_priv), GFP_KERNEL);
	if (!pcm_priv) {
		return -ENOMEM;
	}

#if defined(USE_WORK_Q_FOR_SHIFT_WORK)
	/* note: create_rt_workqueue() API vanished */
	/* Create work-queue */
	pcm_priv->wq = create_workqueue(rtd->dai_link->stream_name);
	if (!(pcm_priv->wq)) {
		goto L_ERR_WQ;
	}
#endif

	spin_lock_init(&pcm_priv->dma_lock);

	/* Set up data structures */
	pcm_priv->card = card;
	pcm_priv->play.pcm = pcm_priv;
	pcm_priv->play.other_stream = &pcm_priv->capt;
	pcm_priv->play.period_bytes_tdm = DMW_PCM_PERIOD_SIZE_TDM;

#if defined(USE_WORK_Q_FOR_SHIFT_WORK)
	INIT_WORK(&pcm_priv->play.shift_work, xfer_shift_work);
#endif

	pcm_priv->capt.pcm = pcm_priv;
	pcm_priv->capt.other_stream = &pcm_priv->play;
	pcm_priv->capt.period_bytes_tdm = DMW_PCM_PERIOD_SIZE_TDM;

#if defined(USE_WORK_Q_FOR_SHIFT_WORK)
	INIT_WORK(&pcm_priv->capt.shift_work, xfer_shift_work);
#endif

	pcm_priv->tdm_owned_by_css = 0;

	/* Get configured DMA channels */
	pcm_priv->play.hw_channel = (unsigned int)cpu_dai->playback_dma_data;
	pcm_priv->capt.hw_channel = (unsigned int)cpu_dai->capture_dma_data;

	/* Pre-allocate DMA buffers : */
	pcm_priv->play.tdm_buffer[0] = kzalloc(DMW_PCM_PERIOD_SIZE_TDM,
						GFP_KERNEL | GFP_DMA);
	if (!(pcm_priv->play.tdm_buffer[0])) {
		goto L_ERR_FREE;
	}
	pcm_priv->play.tdm_buffer[1] = kzalloc(DMW_PCM_PERIOD_SIZE_TDM,
						GFP_KERNEL | GFP_DMA);
	if (!(pcm_priv->play.tdm_buffer[1])) {
		goto L_ERR_FREE;
	}
	pcm_priv->capt.tdm_buffer[0] = kzalloc(DMW_PCM_PERIOD_SIZE_TDM,
						GFP_KERNEL | GFP_DMA);
	if (!(pcm_priv->capt.tdm_buffer[0])) {
		goto L_ERR_FREE;
	}
	pcm_priv->capt.tdm_buffer[1] = kzalloc(DMW_PCM_PERIOD_SIZE_TDM,
						GFP_KERNEL | GFP_DMA);
	if (!(pcm_priv->capt.tdm_buffer[1])) {
		goto L_ERR_FREE;
	}

	pcm_priv->play.dma_state = kDMA_ST_IDLE;
	pcm_priv->capt.dma_state = kDMA_ST_IDLE;


	vdbg_prt("%s %u%s\n",
		 "Initial default config DMA chain, for tdm",
		 cpu_dai->id,
		 ", for Playback");

	ret = tdm_dma_configure(&pcm_priv->play);
	if (ret) {
		goto L_ERR_FREE;
	}

	vdbg_prt("%s %u%s\n",
		 "Initial default config DMA chain, for tdm",
		 cpu_dai->id,
		 ", for Capture");

	ret = tdm_dma_configure(&pcm_priv->capt);
	if (ret) {
		goto L_ERR_FREE;
	}

	/* Save our pcm-private-data on the cpu-dai */
	snd_soc_dai_set_pcm_privdata(cpu_dai, pcm_priv);

	return 0;

L_ERR_FREE:
	/* Free the DMA buffers and the dma list - for each direction */
	dmw_pcm_free_dma_data(&pcm_priv->play);
	dmw_pcm_free_dma_data(&pcm_priv->capt);

#if defined(USE_WORK_Q_FOR_SHIFT_WORK)
L_ERR_WQ:
#endif
	kfree(pcm_priv);
	dev_err(card->dev, "%s failed: %d\n", __FUNCTION__, ret);
	return ret;
}

static void dmw_pcm_free(struct snd_pcm *pcm)
{
	struct snd_soc_pcm_runtime *rtd = pcm->private_data;
	struct dmw_pcm_priv *pcm_priv = snd_soc_dai_get_pcm_privdata(rtd->cpu_dai);

	dbg_prt();

#if defined(USE_WORK_Q_FOR_SHIFT_WORK)
	destroy_workqueue(pcm_priv->wq);
#endif

	/* Free the DMA buffers and the dma list - for each direction */
	dmw_pcm_free_dma_data(&pcm_priv->play);
	dmw_pcm_free_dma_data(&pcm_priv->capt);

	kfree(pcm_priv);
}

static int dmw_soc_suspend(struct snd_soc_dai *dai)
{
	struct dmw_pcm_priv *pcm_priv = snd_soc_dai_get_pcm_privdata(dai);

	dbg_prt("%s %s\n", "for dai", dai->name);

	/* Cancel DMA transferes - in both directions */
	tdm_dma_cancel_xfer(&pcm_priv->play);
	tdm_dma_cancel_xfer(&pcm_priv->capt);

	vdbg_prt("%s %s\n", "Play & Capt DMA xfer canceled for dai",
		 dai->name);

	return 0;
}

static int dmw_soc_resume(struct snd_soc_dai *dai)
{
	//struct dmw_pcm_priv *pcm_priv = snd_soc_dai_get_pcm_privdata(dai);

	dbg_prt("%s %s\n", "for dai", dai->name);

	vdbg_prt("%s\n", "Currently doing nothing ---->");

	return 0;
}

static struct snd_soc_platform_driver dmw_soc_platform = {
	.ops		= &dmw_pcm_ops, 
	.pcm_new	= dmw_pcm_new,
	.pcm_free	= dmw_pcm_free,
	.suspend	= dmw_soc_suspend,
	.resume		= dmw_soc_resume,
};

/*****************************************************************************/

void dmw_pcm_set_telephony(int tel_rqst)
{
	vdbg_prt("%s %s.  %s %s\n",
		 "TEL Request:", (tel_rqst) ? "Tel to ON" : "Tel to OFF",
		 "Previous global TEL state:", (g_tdm_owned_by_css) ? "ON" : "OFF");

	g_tdm_owned_by_css = tel_rqst;

	vdbg_prt("%s %s\n",
		 "Global TEL state set to", (g_tdm_owned_by_css) ? "ON" : "OFF");
}
EXPORT_SYMBOL(dmw_pcm_set_telephony);

/*****************************************************************************/

static int __devinit dmw_pcm_platform_probe(struct platform_device *pdev)
{
	int err;
	
	dbg_prt();

	err = snd_soc_register_platform(&pdev->dev, &dmw_soc_platform);
	if (err) {
		vdbg_prt("%s\n", "snd_soc_register_platform() Failed !!!");
	}
	
	return err;
}

static int __devexit dmw_pcm_platform_remove(struct platform_device *pdev)
{
	dbg_prt();

	snd_soc_unregister_platform(&pdev->dev);
	return 0;
}

static struct platform_driver dmw_pcm_driver = {
	.driver = {
		.name = DRV_NAME,
		.owner = THIS_MODULE,
	},
	.probe = dmw_pcm_platform_probe,
	.remove = __devexit_p(dmw_pcm_platform_remove),
};

/*****************************************************************************/

static int __init snd_dmw_pcm_init(void)
{
	int err;
#ifdef DBG_DMA_GPIOs
	uint i;
#endif

	dbg_prt();

#ifdef DBG_DMA_GPIOs
	vdbg_prt("%s\n", "Enabling debug GPIOs");
	for (i = 0; i < ARRAY_SIZE(g_dbg_gpio); i++) {
		err = gpio_request(g_dbg_gpio[i], "dmw-pcm");
		if (err < 0) {
			vdbg_prt("%s\n", "Can't get the debug gpio");
			return err;
		}
		gpio_direction_output(g_dbg_gpio[i], 0);
	}
#endif /* DBG_DMA_GPIOs */

	vdbg_prt("%s\n", "Registrating platform driver");
	err = platform_driver_register(&dmw_pcm_driver);
	if (err) {
		vdbg_prt("%s\n", "platform_driver_register() Failed !!!");
	}
	return err;
}
module_init(snd_dmw_pcm_init);

static void __exit snd_dmw_pcm_exit(void)
{
#ifdef DBG_DMA_GPIOs
	uint i;
#endif
	
	dbg_prt();

#ifdef DBG_DMA_GPIOs
	vdbg_prt("%s\n", "Disabling Debug GPIOs");
	for (i = 0; i < ARRAY_SIZE(g_dbg_gpio); i++) {
		gpio_free(g_dbg_gpio[i]);
	}
#endif /* DBG_DMA_GPIOs */

	vdbg_prt("%s\n", "Unregistrating platform driver");
	platform_driver_unregister(&dmw_pcm_driver);
}
module_exit(snd_dmw_pcm_exit);

MODULE_DESCRIPTION("DMW ASoC platform DMA driver");
MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:" DRV_NAME);

