
#  Basic editing  #
# backSpace - delete selection, or a single char if no selection.
proc backSpace {} {
    set sel [selectLimits]
    if {$sel != ""} {
	eval text_cmd delete $sel
    } else {
	text_cmd delete "insert -1c"
    }
}
# backwardChar - moves insertion one char back
proc backwardChar {args} {
    text_cmd mark set insert "insert-1c"
}
# backwardCharSelect - extends selection one char back
proc backwardCharSelect {} {
    set sel [selectLimits]
    if {$sel == ""} { set p [getPos] } else { set p [lindex $sel 0] }
    text_cmd tag add sel "$p -1c" $p
}
# backwardDeleteWord - deletes previous word
proc backwardDeleteWord {} {
    deleteText [text_cmd backward_word insert] insert
}
# backwardWord - moves insertion one word back
proc backwardWord {} {
    goto [text_cmd backward_word insert]
}
# forwardWord - move insertion one word forward
proc forwardWord {} {
    goto [text_cmd forward_word insert]
}

# beginningBufferSelect - extend selection to the 
#  beginning of the buffer
proc beginningBufferSelect {} {
    text_cmd tag add sel 1.0 insert
}
# beginningLineSelect - extend selection to the 
#  beginning of the line
proc beginningLineSelect {} {
    text_cmd tag add sel "insert linestart" insert
}
# beginningOfBuffer - move insertion to the beginning 
#  of the buffer
proc beginningOfBuffer {} {
    text_cmd mark set insert 1.0
}
# beginningOfLine - move insertion to the beginning of 
#  the line
proc beginningOfLine {} {
    text_cmd mark set insert "insert linestart"
}
# blink <pos> - blink cursor at 'pos'
proc blink {pos} {
    text_cmd tag add blink $pos
    getWinInfo w
    if {[info exists w(currline)]} {
	set topl $w(currline)
	set endl [expr {$topl + $w(linesdisp)}]
	scan [posToRowCol $pos] "%d %d" row col
	if {$row < $topl || $row >= $endl} {
	    message "Matching '[getText [lineStart $pos] [pos::math $pos + 1]]'"
	}
    }
    after 500 [list text_cmd tag remove blink $pos]
}
# capitalizeRegion - capitalize all words in selected 
#  region 
proc capitalizeRegion {args} {echo "capitalizeRegion $args"}
# capitalizeWord - capitalize word
proc capitalizeWord {args} {echo "capitalizeWord $args"}
# centerRedraw - redraw window with current line in 
#  the middle.
proc centerRedraw {args} {
    global win::tk
    getWinInfo w
    set topl $w(currline)
    set endl [expr {$topl + $w(linesdisp)}]
    foreach {row col} [posToRowCol [getPos]] {}
    if {$row < $topl - 10 || $row >= 10 + $endl} {
	eval $win::tk([win::Current]) see [getPos]
    } else {
	eval $win::tk([win::Current]) yview scroll [expr {$row - $topl - $w(linesdisp)/2}] units
    }
}
# clear - clear selected text
proc clear {} {
    global win::tk
    eval $win::tk([win::Current]) delete [selectLimits]
}
# copy - copy region
proc copy {} {
    global win::tk
    tk_textCopy $win::tk([win::Current])
    #event generate [focus] <<Copy>>
}
# paste - insert the last chunk of text created by 'cut' 
#  or 'copy'
proc paste {} {
    global win::tk
    catch {eval $win::tk([win::Current]) delete [selectLimits]}
    tk_textPaste $win::tk([win::Current])
    #event generate [focus] <<Paste>>
}
# cut - deletes and saves region
proc cut {args} {
    global win::tk
    tk_textCut $win::tk([win::Current])
}
# createTagFile - searches all files in current file set 
#  and saves the locations of any function declarations
#  in a file called 'cTAGS'.
proc createTagFile {args} {echo "createTagFile $args"}
# deleteChar - delete char AFTER cursor
proc deleteChar {} {
    set sel [selectLimits]
    if {$sel != ""} {
	eval text_cmd delete $sel
    } else {
	text_cmd delete "insert"
    }
}
# deleteSelection - delete current position, don't save
proc deleteSelection {} {
    eval text_cmd delete [selectLimits]
}
# deleteWord - delete word after cursor
proc deleteWord {} {
    text_cmd delete insert "insert wordend"
}
# deleteText <pos1> <pos2> - remove text between 'pos1' 
#  and 'pos2'
proc deleteText {pos1 pos2} {
    text_cmd delete $pos1 $pos2
}
# downcaseRegion - changes all uppercase letters to 
#  lowercase in current region
proc downcaseRegion {} {
    eval replaceText [selectLimits] [list [string tolower [eval getText [selectLimits]]]]
}
# downcaseWord - changes all uppercase letters to 
#  lowercase in current word
proc downcaseWord {} {
    set start [text_cmd backward_word insert]
    set end [text_cmd forward_word insert]
    replaceText $start $end [string toupper [getText $start $end]]
}
# endBufferSelect - extend selection to the end of the 
#  buffer
proc endBufferSelect {} {
    text_cmd tag add sel insert end
}
# endLineSelect - extend selection to the end of line
proc endLineSelect {} {
    text_cmd tag add sel insert "insert lineend"
}
# endOfBuffer - move insertion to the end of the buffer
proc endOfBuffer {} {
    text_cmd mark set insert "end"
}
# endOfLine - move insertion to the end of the line
proc endOfLine {} {
    text_cmd mark set insert "insert lineend"
}
proc enterReplaceString {} {replaceString [getSelect]}
proc enterSearchString {} {searchString [getSelect]}
# exchangePointAndMark - exchange the current 'mark' 
#  w/ the current insertion point
proc exchangePointAndMark {args} {echo "exchangePointAndMark $args"}
# forwardChar - move insertion one character forward
proc forwardChar {} {
    text_cmd mark set insert "insert +1c"
}
# forwardCharSelect - extend selection one character 
#  forward
proc forwardCharSelect {} {
    set sel [selectLimits]
    if {$sel == ""} { set p [getPos] } else { set p [lindex $sel 1] }
    text_cmd tag add sel $p "$p +1c"
}
# getMark - return the current mark.
proc getMark {args} {echo "getMark $args"}
# getPos [-w <win>] - return the current insertion point
# By default Alpha and Tk manipulate selections differently.  If you
# select some text with the cursor/mouse, Alpha leaves the insertion point
# at the start of the selction.  Tk leaves it wherever you last left it --- 
# often at the end or middle of the selection.  The best solution is really
# to change the default behaviour encoded in text.tcl, but until that is
# fixed, we really need this proc to be more than just 'text_cmd index insert'.
proc getPos {args} {
    set sel [selectLimits]
    if {[llength $sel]} {
	return [lindex $sel 0]
    } else {
	text_cmd index insert
    }
}
# putScrap [<string>]+ - Concatenate strings together into the system 
#  scrap. The scrap can be appended to through calls of the form 'putScrap 
#  [getScrap] " another word"'.
proc putScrap {args} {
    clipboard clear
    eval clipboard append $args
}
# getScrap - returns system TEXT scrap.
proc getScrap {} {
    selection get -selection CLIPBOARD
}
# getSelect - return the currently selected text, if 
#  any.
proc getSelect {} {
    set sel [selectLimits]
    if {[llength $sel]} {return [eval text_cmd get $sel]} else {return ""}
}

# getText [-w <win>] <pos1> <pos2> - return the text between 'pos1' 
#  and 'pos2'. '-w' can be used to specify a window.
proc getText {pos1 pos2 args} {
    if {$pos1 == "-w"} {
	text_wcmd $pos2 get [lindex $args 0] [lindex $args 1]
    } else {
	text_cmd get $pos1 $pos2
    }
    
}

# getNamedMarks [-w <win>] [-n] - 
#  return list of all permanent marks in open files. Each
#  element of a list describes one mark as a sublist of the mark's name, 
#  the complete pathname of the mark's file, the position of the first 
#  character in the first line displayed, the current position, and the 
#  end of the selection if text is hilited, or the current position again 
#  if there is no hilited section. '-w' allows window name to be applied 
#  as filter, '-n' means only names will be returned.
proc getNamedMarks {args} {
    getOpts {-w}
    if {[info exists opts(-w)]} {
	set w $opts(-w)
    } else {
	set w [win::Current]
    }
    global win::tk
    set w $win::tk($w)
    if {[info exists opts(-n)]} {
	set res ""
	foreach m [$w tag names] {
	    if {[regexp "^mark:(.*)$" $m "" mm]} {
		lappend res $mm
	    }
	}
	return $res
    } else {
	set res ""
	foreach m [$w tag names] {
	    if {[regexp "^mark:(.*)$" $m "" mm]} {
		set r [$w tag ranges $m]
		lappend res [eval list [list $mm "" [lindex $r 0]] $r]
	    }
	}
	return $res
    }
}
# setNamedMark [name disp pos end] - set named mark. If optional arguments are 
#  present, the mark is created without prompting user. 'disp' is the 
#  character position of the start of the first line to be displayed, 
#  while 'pos' and 'end' bracket the text to be selected.
proc setNamedMark {name disp pos end} {
    set pos [text_cmd index $pos]
    set end [text_cmd index $end]
    if {$pos == $end} { set end "$end +1c" }
    #echo "$name $pos $end"
    # we'll ignore disp
    text_cmd tag add "mark:$name" $pos $end
}
# removeMark [-all]|[[-n <mark name] [-w <specname>]]- allows marks to be 
#  removed. If no options are specified, it's interactive.
proc removeMark {args} {uplevel 1 removeNamedMark $args}

proc removeNamedMark {args} {
    getOpts {-n -w}
    if {[info exists opts(-w)]} {
	set w $opts(-w)
    } else {
	set w [win::Current]
    }
    if {[info exists opts(-all)]} {
	foreach m [text_wcmd $w tag names] {
	    if {[regexp "^mark:" $m]} {
		text_wcmd $w tag delete $m
	    }
	}
	return
    }
    text_wcmd $w tag delete "mark:$opts(-n)"
}

# getTMarks - Return a list of temporary marks. Each item of the returned 
#  list is a sublist containing the mark name, the complete pathname of the 
#  mark, and the start and finish of the selection named by the mark. The 
#  following is an example of the result: 
#
#    {{temp1 External:file.c 1312 1315} {temp2 Internal:it.h 111 111}} 
#
proc getTMarks {} {
    global win::Current
    foreach m [text_cmd mark names] {
	set where [text_cmd index $m]
	lappend res [list $m $win::Current $where $where]
    }
    return $res
}
namespace eval tmark {}

proc tmark::getPos {m} {
    text_cmd index $m
}

proc tmark::isAt {p} {
    set m [text_cmd mark next $p]
    if {[text_cmd compare $m == $p]} {
	return $m
    } else {
	return ""
    }
}

proc tmark::getPositions {mm} {
    foreach m $mm {
	lappend res [text_cmd index $m]
    }
    return $res
}

proc tmark::getRange {m {f ""}} {
    if {$f != ""} {
	set res [text_wcmd $f tag ranges $m]
    } else {
	set res [text_cmd tag ranges $m]
    }
    if {$res == ""} {
	error "No such mark"
    }
    return [list [lindex $res 0] "" [lindex $res 1]]
}

namespace eval mark {}
proc mark::getRange {m {f ""}} {
    if {$f != ""} {
	set res [text_wcmd $f tag ranges mark:$m]
    } else {
	set res [text_cmd tag ranges mark:$m]
    }
    if {$res == ""} {
	error "No such mark"
    }
    return [list [lindex $res 0] "" [lindex $res 1]]
}

proc refresh {args} {}

# display [-w <win>] <pos> - move pos's line to top of screen.
proc display {pos args} {
    if {$pos == "-w"} {
	text_wcmd [lindex $args 0] see [lindex $args 1]
	global win::tk
	raise [winfo parent $win::tk([lindex $args 0])]
    } else {
	text_cmd see $pos
    }
}


# goto <pos> - goto the position 'pos'.
proc goto {pos args} {
    if {$pos == "-w"} {
	text_wcmds [lindex $args 0] "mark set insert [lindex $args 1]" "see insert"
	select -w [lindex $args 0] insert insert
    } else {
	text_cmds "mark set insert [list $pos]" "see insert"
	select insert insert
    }
}

proc compare {i op i2} {
    text_cmd compare $i $op $i2
}

# gotoMark - goto named mark, use 'mark' in macros.
proc gotoMark {mark} {
    text_cmds [list mark set insert mark:${mark}.first] "see insert"    
    eval select [text_cmd tag ranges mark:$mark]
}
# createTMark <name> <pos> - create a temporary 'mark' 
#  at location 'pos'. 
proc createTMark {name pos} {
    if {[string first "." $pos] != -1} {
	text_cmds [list mark set $name $pos] [list mark gravity $name left]
    } else {
	# old format
	text_cmd mark set $name "1.0 +${pos}c"
    }
}
# removeTMark <name> - remove temporary mark.
proc removeTMark {name} {
    text_cmd mark unset $name
}
# gotoTMark <name> - goto the temporary mark 'name'.
proc gotoTMark {mark} {
    text_cmds "mark set insert $mark" "see insert"  
    select insert insert
}
# insertText [-w <win name>] <text>* - Insert 'text' at the current 
#  insertion point.
proc insertText {args} {
    global win::tk
    if {[lindex $args 0] == "-w"} {
	set w $win::tk([lindex $args 1])
	set text [join [lrange $args 2 end] ""]
	regsub -all "\r" $text "\n" text
	$w insert insert $text
    } else {
	global win::Active
	if {[info exists win::Active]} {
	    set w $win::tk([lindex $win::Active 0])
	    set text [join $args ""]
	    regsub -all "\r" $text "\n" text
	    $w insert insert $text
	} else {
	    error "No window!"
	}
    }
}

# insertToTop - make the line that the insertion point 
#  is on the first line shown, and display the current 
#  line number along w/ the total number of lines in file
proc insertToTop {} {
    text_cmd see insert
}
# killLine - kill text from insertion point to the end 
#  of the line. If the line has no text, delete the line 
#  and move succeeding lines up one.
proc killLine {} {
    text_cmd delete insert "insert lineend"
}
# lineStart <pos> - return the position of the start of
#  the line 'pos' is on.
proc lineStart {m} {return "$m linestart"}
# lookAt [-w <name>] <pos> - return the 'pos'th character of the 
#  current file, or the file named by <name> if the '-w' option is specified.
proc lookAt {pos} {
    return [text_cmd get $pos]
}
# markHilite - This is the 'Hilite' from the 'Edit'
#  menu. If there is a currently hilited selection, the 
#  selection is unhilited, leaving the mark and the 
#  insertion point around the old selection. If there 
#  is not a selection, the region between the insertion 
#  point and the mark is selected.
proc markHilite {args} {echo "markHilite $args"}
# matchBrace - moves the insertion point to the 
#  character that matches the character after the current 
#  insertion point
proc matchBrace {} {
    goto [matchIt [text_cmd get insert] insert]
}
# balance - selects smallest set of parens, braces, or 
#  brackets that encloses the current selection
proc balance {} {
    text_cmd balance
}
# matchIt <brace char> <pos> [<limit>] - Return pos of matching brace. Recognizes 
#  parenthesis, square brackets, and curly braces. Optional third argument 
#  specifies how many characters to search.
proc matchIt {args} {
    return [eval text_cmd match $args]
}
# Returns minimum position in the window (depends upon window system
# whether this starts with 1 or 0 or ...)
proc minPos {} { return 1.0 }
# maxPos [-w <win>] - returns the number of characters in the front
#  window.
proc maxPos {args} {
    if {[lindex $args 0] == "-w"} {
	return [text_wcmd [lindex $args 1] index "end -1c"]
    } else {
	return [text_cmd index "end -1c"]
    }
}
# moveInsertionHere [-last] - move the insertion point to the 
#  first (or last) line displayed
proc moveInsertionHere {{where -first}} {
    getWinInfo w
    set topl $w(currline)
    foreach {row col} [posToRowCol [getPos]] {}
    if {$where == "-first"} {
	goto "[expr {$topl +1}].0"
    } else {
	set endl [expr {$topl + $w(linesdisp)}]
	goto "$endl.0"
    }
}

# nextLine - move insertion point to next line
proc nextLine {} {
    goto "insert +1l"
}
# nextLineSelect - extend selection to the next line
proc nextLineSelect {} {
    set sel [selectLimits]
    if {$sel == ""} {
	set p [getPos]
    } else {
	set p [lindex $sel 1]
    }
    text_cmd tag add sel $p "$p +1l"
}
# nextLineStart <pos> - return the position of the start 
#  of the next line after position 'pos'.
proc nextLineStart {m} {return "$m linestart +1l"}
# nextSentence - go to next sentence ("fill.tcl").
proc nextSentence {args} {echo "nextSentence $args"}
# oneSpace - converts whitespace surrounding insertion
#  into a single space.
proc oneSpace {} {
    set p [getPos]
    set first [lindex [search -s -n -r 1 -f 0 {[^ \t\r\n]} $p] 1]
    set last [lindex [search -s -n -r 1 -f 1 {[^ \t\r\n]} $p] 0]
    if {$first == ""} {
	set first [minPos]
    }
    if {$last == ""} {
	set last [maxPos]
    }
    replaceText $first $last " "
}
# openLine - insert a new line following the current 
#  one and move the insertion point to it
proc openLine {args} {
    goto "insert lineend"
    insertText "\r"
}
# pageBack - display prev screenful, move the
#  insertion point if 'moveInsertion' enabled
proc pageBack {args} {
    global win::tk
    eval $win::tk([win::Current]) yview scroll -1 pages
}
# pageForward - display next screenful, move the
#  insertion point if 'moveInsertion' enabled
proc pageForward {args} {
    global win::tk
    eval $win::tk([win::Current]) yview scroll 1 pages
}
# posToRowCol <pos> - converts from absolute position to row, col.
proc posToRowCol {pos} {
    if {[catch {text_cmd index $pos} p]} {
	set row [lindex [split [text_cmd index 1.0+${pos}c] .] 0]
    } else {
	set row [lindex [split $p .] 0]
    }
    set col [string length [text::maxSpaceForm [getText [lineStart $pos] $pos]]]
    return [list $row $col]
}
# previousLine - move insertion point to the previous 
#  line
proc previousLine {} {
    goto "insert -1l"
}
# prevLineSelect - extend selection to the previous line
proc prevLineSelect {} {
    set sel [selectLimits]
    if {$sel == ""} {
	set p insert
    } else {
	set p [lindex $sel 0]
    }
    text_cmd tag add sel "$p -1l" $p
}
# prevSentence - go to previous sentence ("fill.tcl").
proc prevSentence {args} {echo "prevSentence $args"}
# rectMarkHilite - creates a rectangular selection 
#  between the mark and the insertion point.
proc rectMarkHilite {args} {echo "rectMarkHilite $args"}
# replaceString [<str>] - specifies replacement string. Returns current 
#  replacement string if no arg.
proc replaceString {args} {
    global replaceString
    if {[llength $args] == 0} {
	return $replaceString
    } else {
	set replaceString [lindex $args 0]
    }
}
# replaceText <pos1> <pos2> [text]+ - replaces the text
#  between 'pos1' and 'pos2' with 'text', where 'text' can be any number 
#  of arguments. Leaves insertion at end, mark at beginning of inserted 
#  text. 
proc replaceText {begin end args} {
    deleteText $begin $end
    text_cmd mark set insert $begin
    eval insertText $args
}
# rowColToPos [-w <win>] <row> <col> - converts to abosolute position.
#  Accepts optional -w parameter that allows window to be specified.
proc rowColToPos {row col args} {
    if {$row == "-w"} {
	return "[lindex $args 0].[lindex $args 1]"
    } else {
	return $row.$col
    }
}
# scrollDownLine - same action as that which occurs when 
#  the down arrow in the vertical scrollbar is selected
proc scrollDownLine {args} {
    global win::tk
    eval $win::tk([win::Current]) yview scroll 1 units
}

# scrollLeftCol - same action as that which occurs when
#  the left arrow in the horizontal scrollbar is selected
proc scrollLeftCol {chars} {
    global win::tk
    eval $win::tk([win::Current]) xview scroll -$chars units
}
# scrollRightCol - same action as that which occurs when
#  the right arrow in the horizontal scrollbar is 
#  selected
proc scrollRightCol {chars} {
    global win::tk
    eval $win::tk([win::Current]) xview scroll $chars units
}
# scrollUpLine - same action as that which occurs when 
#  the up arrow in the vertical scrollbar is selected
proc scrollUpLine {args} {
    global win::tk
    eval $win::tk([win::Current]) yview scroll -1 units
}
# select [-w <win>] <pos1> <pos2> - selects the text between 'pos1' 
#  and 'pos2'.
proc select {pos1 {pos2 ""} args} {
    global win::tk
    if {$pos1 == "-w"} {
	set w $pos2
	set pos1 [lindex $args 0]
	set pos2 [lindex $args 1]
	set cmd [list text_wcmd $w]
	set cmds [list text_wcmds $w]
	raise [winfo parent $win::tk($w)]
    } else {
	set cmd text_cmd
	set cmds text_cmds
	set w [win::Current]
    }
    
    if {$pos2 == ""} { set pos2 $pos1 }
    if {[pos::compare $pos1 > $pos2]} {
	set p $pos1
	set pos1 $pos2
	set pos2 $p
	unset p
    }
    
    eval $cmd mark set insert [list $pos1]
    # 99% of the time we know whether we should be removing
    # sel or backsel, but sometimes when the window is in
    # the process of receiving the focus we can be wrong.
    # Hence we clear both.
    set range [eval $cmd tag ranges sel]
    if {$range != ""} {
	eval $cmd tag remove sel $range
    }
    set range [eval $cmd tag ranges backsel]
    if {$range != ""} {
	eval $cmd tag remove backsel $range
    }
    
    if {[string match "[winfo parent $win::tk($w)].*" [focus]]} {
	eval $cmds [list [list tag add sel $pos1 $pos2] [list see $pos1]]
    } else {
	eval $cmds [list [list tag add backsel $pos1 $pos2] [list see $pos1]]
    }
}

# selEnd [-w <win] - returns the end of the hilited selection, or 
#  the current insertion point if no text is selected.
proc selEnd {args} {
    set sel [selectLimits]
    if {$sel != ""} {
	return [lindex $sel 1]
    } else {
	return [text_cmd index insert]
    }
}
# setMark - set the current mark to the insertion point
proc setMark {} {
    text_cmd mark set sel insert
}
# sortMarks - sorts all marks of open windows, w/o 
#  setting the dirty flag. If the files are saved, the 
#  marks will be saved in alphabetical order.
proc sortMarks {args} {echo "sortMarks $args"}
# spacesToTabs - convert spaces in selection to tabs
proc spacesToTabs {{start ""} {end ""}} {
    if {$start == ""} {
	set start [getPos]
	set end [selEnd]
    }
    set from [lindex [posToRowCol $start] 0]
    set to [lindex [posToRowCol $end] 0]
    while {$from <= $to} {
	set pos [rowColToPos $from 0]
	# get the leading whitespace of the current line
	set res [search -s -n -f 1 -r 1 "^\[ \t\]*" $pos]
	if {![llength $res]} {
	    # end of the file
	    return
	}
	set front [text::minSpaceForm [eval getText $res]]
	eval replaceText $res [list $front]
	incr from
    }
}
# tab - insert a tab
proc tab {} {typeText "\t"}
# tabsToSpaces - convert tabs in selection to spaces.
proc tabsToSpaces {{start ""} {end ""}} {
    if {$start == ""} {
	set start [getPos]
	set end [selEnd]
    }
    set from [lindex [posToRowCol $start] 0]
    set to [lindex [posToRowCol $end] 0]
    while {$from <= $to} {
	set pos [rowColToPos $from 0]
	# get the leading whitespace of the current line
	set res [search -s -n -f 1 -r 1 "^\[ \t\]*" $pos]
	if {![llength $res]} {
	    # end of the file
	    return
	}
	set front [text::maxSpaceForm [eval getText $res]]
	eval replaceText $res [list $front]
	incr from
    }
}
# upcaseRegion - convert all lowercase letters to 
#  uppercase in the current region
proc upcaseRegion {} {
    eval replaceText [selectLimits] [list [string toupper [eval getText [selectLimits]]]]
}
# upcaseWord - convert all lowercase letters to 
#  uppercase in the current word
proc upcaseWord {} {
    set start [text_cmd backward_word insert]
    set end [text_cmd forward_word insert]
    replaceText $start $end [string toupper [getText $start $end]]
}
# wrap - see the section on "Fill" routines.
proc wrap {args} {echo "wrap $args"}
# wrapText
proc wrapText {args} {echo "wrapText $args"}
# yank - insert the last piece of deleted text of less
#  than 1k. Consecutive deletes are concatenated.
#  together.
proc yank {args} {echo "yank $args"}
